<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use Illuminate\Http\Request;
use App\Models\Member;

class AttendanceController extends Controller
{
    // GET: Fetch all attendance records
    public function index()
    {
        $attendances = Attendance::all();
        return response()->json($attendances);
    }

    // POST: Create a new attendance record
    public function store(Request $request)
    {
        $request->validate([
            'date' => 'required|date',
            'member_id' => 'required|exists:members,id',
            'sabha_id' => 'required|exists:sabhas,id',
        ]);

        // Check for duplicate
        $existing = Attendance::where('member_id', $request->member_id)
            ->where('sabha_id', $request->sabha_id)
            ->where('date', $request->date)
            ->first();

        if ($existing) {
            return back()->with([
                'message' => 'Attendance already marked for this member, sabha, and date.',
                'alert-type' => 'error'
            ]);
        }

        // Create
        $attendance = Attendance::create($request->only('date', 'member_id', 'sabha_id'));

        return redirect()
            ->route("voyager.attendances.index")
            ->with([
                'message' => 'Attendance added successfully.',
                'alert-type' => 'success'
            ]);
    }

    // GET: Fetch a specific attendance record
    public function show($id)
    {
        $attendance = Attendance::with('memberDetails')->find($id);

        if (!$attendance) {
            return response()->json(['message' => 'Attendance not found'], 404);
        }

        return response()->json($attendance);
    }

    // PUT: Update an existing attendance record
    public function update(Request $request, $id)
    {
        $request->validate([
            'date' => 'nullable|date',
            'member_id' => 'required|exists:members,id',
        ]);

        $attendance = Attendance::find($id);

        if (!$attendance) {
            return response()->json(['message' => 'Attendance not found'], 404);
        }

        $attendance->update($request->all());

        return response()->json(['message' => 'Attendance updated successfully', 'attendance' => $attendance]);
    }

    // DELETE: Remove an attendance record
    public function destroy($id)
    {
        $attendance = Attendance::find($id);

        if (!$attendance) {
            return response()->json(['message' => 'Attendance not found'], 404);
        }

        $attendance->delete();

        return response()->json(['message' => 'Attendance deleted successfully']);
    }

    public function markAttendance(Request $request)
    {
        // Validate request
        $request->validate([
            'member_id' => 'required|exists:members,id',
            'sabha_id' => 'required|exists:sabhas,id', // Add this line if you have a sabhas table
        ]);

        // Get today's date
        $today = now()->toDateString();

        // Check if attendance is already recorded
        $attendance = Attendance::where('member_id', $request->member_id)
            ->where('sabha_id', $request->sabha_id) // Add this line to check by sabha_id too
            ->whereDate('date', $today)
            ->first();

        if ($attendance) {
            return response()->json([
                'message' => 'Attendance already marked',
                'attendance' => $attendance
            ], 200);
        }

        // Create new attendance
        $newAttendance = Attendance::create([
            'member_id' => $request->member_id,
            'sabha_id' => $request->sabha_id, // Add this line
            'date' => $today,
        ]);

        return response()->json([
            'message' => 'Attendance marked successfully',
            'attendance' => $newAttendance
        ], 201);
    }

    // public function search(Request $request)
    // {
    //     $search = $request->input('search');

    //     $members = \App\Models\Member::with('mandalDetails')
    //         ->where(function ($query) use ($search) {
    //             $query->whereRaw('CAST(id AS CHAR) LIKE ?', ["%{$search}%"])
    //                 ->orWhere('number', 'LIKE', "%{$search}%");
    //         })
    //         ->orderByRaw("CASE WHEN id = ? THEN 0 ELSE 1 END", [(int) $search]) // exact match on top
    //         ->limit(20)
    //         ->get()
    //         ->map(function ($member) {
    //             return [
    //                 'id' => $member->id,
    //                 'first_name' => $member->first_name,
    //                 'last_name' => $member->last_name,
    //                 'number' => $member->number,
    //                 'mandal_details' => $member->mandalDetails 
    //                     ? ['title' => $member->mandalDetails->title] 
    //                     : null
    //             ];
    //         });

    //     if ($members->isEmpty()) {
    //         return response()->json(['error' => 'Member not found'], 404);
    //     }

    //     return response()->json($members);
    // }

public function search(Request $request)
{
    $search = $request->input('search');

    $members = \App\Models\Member::with('mandalDetails')
        ->where(function ($query) use ($search) {
            $query->whereRaw('CAST(id AS CHAR) LIKE ?', ["%{$search}%"])
                  ->orWhere('number', 'LIKE', "%{$search}%")
                  ->orWhere('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%");
        })
        ->orderByRaw("CASE WHEN id = ? THEN 0 ELSE 1 END", [(int) $search]) // exact ID match on top
        ->limit(20)
        ->get()
        ->map(function ($member) {
            return [
                'id' => $member->id,
                'first_name' => $member->first_name,
                'last_name' => $member->last_name,
                'number' => $member->number,
                'mandal_details' => $member->mandalDetails 
                    ? ['title' => $member->mandalDetails->title] 
                    : null
            ];
        });

    if ($members->isEmpty()) {
        return response()->json(['error' => 'Member not found'], 404);
    }

    return response()->json($members);
}





    public function getMember($id)
    {
        $member = Member::with('mandalDetails')->find($id);

        if (!$member) {
            return response()->json(['error' => 'Member not found'], 404);
        }

        return response()->json([
            'id' => $member->id,
            'first_name' => $member->first_name,
            'last_name' => $member->last_name,
            'mandal_details' => $member->mandalDetails
        ]);
    }
}
