<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use TCG\Voyager\Models\Member;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade as PDF;

class MemberController extends Controller
{
    // GET: Fetch all members
    public function index()
    {
        $members = Member::all();
        return response()->json($members);
    }

    // POST: Add a new member
    public function store(Request $request)
    {
        try {
            $request->validate([
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'member_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
                'number' => 'required|string|regex:/^\d{10}$/', // Only 10 digits without country code
                'gender' => 'nullable|string|max:20',
                'dob' => 'nullable|date',
                'age' => 'nullable|integer',
                'address' => 'nullable|string',
                'city' => 'nullable|string',
                'mandal' => 'nullable|integer',
                'status' => 'nullable|integer',
                'Karyakarta' => 'nullable|string|max:255',
            ], [
                'first_name.required' => 'The first name field is required.',
                'number.required' => 'The phone number field is required.'
            ]);
    
            // Prepare data for insertion
            $memberData = $request->all();
    
            // Ensure the number starts with +91
            $phone_number = $request->input('number');
            if (substr($phone_number, 0, 3) !== '+91') {
                $phone_number = '+91' . $phone_number;
            }
            $memberData['number'] = $phone_number;  // Update the number in member data
    
            // Check if image is present
            if ($request->hasFile('member_image')) {
                // Handle the image upload
                $imagePath = $request->file('member_image')->store('member_images', 'public');
                $memberData['member_image'] = $imagePath;
            } else {
                // Default image if none is provided
                $memberData['member_image'] = 'member_images/default.png';
            }
    
            // Create member
            $member = Member::create($memberData);
    
            return response()->json(['message' => 'Member created successfully', 'member' => $member], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
    }
    
    public function update(Request $request, $id)
    {
        try {
            $member = Member::find($id);
    
            if (!$member) {
                return response()->json(['message' => 'Member not found'], 404);
            }
    
            // Validation for the request
            $request->validate([
                'first_name' => 'nullable|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'member_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
                'number' => 'nullable|string|regex:/^\d{10}$/',
                'gender' => 'nullable|string|max:20',
                'dob' => 'nullable|date',
                'age' => 'nullable|integer',
                'address' => 'nullable|string',
                'city' => 'nullable|string',
                'mandal' => 'nullable|integer',
                'status' => 'nullable|integer',
                'Karyakarta' => 'nullable|string|max:255',
            ]);
    
            // Handle the phone number to add +91 if not present
            if ($request->has('number')) {
                $phone_number = $request->input('number');
                if (substr($phone_number, 0, 3) !== '+91') {
                    $phone_number = '+91' . $phone_number;
                }
                $request->merge(['number' => $phone_number]);
            }
    
            // Update member data with new values (excluding member_image)
            $member->update($request->except('member_image'));
    
            // Handle file upload if a new image is provided
            if ($request->hasFile('member_image')) {
                $imagePath = $request->file('member_image')->store('member_images', 'public'); // Store image in `storage/app/public/member_images`
                $member->member_image = $imagePath;  // Save the new image path in the member record
                $member->save(); // **Save the model to persist the image path**
            }
    
            return response()->json(['message' => 'Member updated successfully', 'member' => $member]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred.'], 500);
        }
    }
    
    // DELETE: Delete a member
    public function destroy($id)
    {
        $member = Member::find($id);

        if (!$member) {
            return response()->json(['message' => 'Member not found'], 404);
        }

        $member->delete();

        return response()->json(['message' => 'Member deleted successfully']);
    }

    public function generateCard($id)
    {
        $member = Member::findOrFail($id);
        return view('admin.members.card', compact('member'));
    }
    // Helper function to save Base64 image
    private function saveBase64Image($base64String)
    {
        // Validate Base64 format
        if (!preg_match('/^data:image\/(\w+);base64,/', $base64String, $matches)) {
            return null;
        }

        $imageType = $matches[1]; // jpg, png, gif, etc.
        $imageData = substr($base64String, strpos($base64String, ',') + 1);
        $imageData = base64_decode($imageData);

        // Generate unique filename
        $filename = 'member_images/' . Str::random(20) . '.' . $imageType;

        // Store in Laravel storage (storage/app/public/member_images)
        Storage::disk('public')->put($filename, $imageData);

        return $filename; // Save path in database
    }
}
