<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use TCG\Voyager\Models\Attendance;
use Illuminate\Http\Request;
use App\Models\Sabha;
use Illuminate\Validation\ValidationException;

class AttendanceController extends Controller
{
    // GET: Fetch all attendance records
    public function index()
    {
        $attendances = Attendance::with(['memberDetails', 'sabha'])->get();

        return response()->json(['data' => $attendances], 200);
    }

    // POST: Create a new attendance record
    public function store(Request $request)
    {
        try {
            $request->validate([
                'member_id' => 'required|exists:members,id',
                'sabha_id'  => 'required|exists:sabhas,id',
            ]);

            $today = now()->toDateString();

            // Prevent duplicate entry
            $existing = Attendance::where('member_id', $request->member_id)
                ->where('sabha_id', $request->sabha_id)
                ->whereDate('date', $today)
                ->first();

            if ($existing) {
                return response()->json([
                    'message' => 'Attendance already exists for this member, sabha, and date.',
                    'attendance' => $existing
                ], 409);
            }

            $attendance = Attendance::create([
                'date'      => $today,
                'member_id' => $request->member_id,
                'sabha_id'  => $request->sabha_id,
            ]);

            return response()->json([
                'message' => 'Attendance created successfully',
                'attendance' => $attendance
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Server error',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // GET: Fetch a specific attendance record
    public function show($id)
    {
        $attendance = Attendance::with(['memberDetails', 'sabha'])->find($id);

        if (!$attendance) {
            return response()->json(['message' => 'Attendance not found'], 404);
        }

        return response()->json(['data' => $attendance], 200);
    }

    // PUT: Update an existing attendance record
    public function update(Request $request, $id)
    {
        try {
            $request->validate([
                'date'       => 'nullable|date',
                'member_id'  => 'required|exists:members,id',
                'sabha_id'   => 'required|exists:sabhas,id',
            ]);

            $attendance = Attendance::find($id);

            if (!$attendance) {
                return response()->json(['message' => 'Attendance not found'], 404);
            }

            $exists = Attendance::where('member_id', $request->member_id)
                ->where('sabha_id', $request->sabha_id)
                ->whereDate('date', $request->date ?? $attendance->date)
                ->where('id', '!=', $id)
                ->first();

            if ($exists) {
                return response()->json([
                    'message' => 'Duplicate attendance exists for this member, sabha, and date.',
                    'conflict_id' => $exists->id
                ], 409);
            }

            $attendance->update([
                'member_id' => $request->member_id,
                'sabha_id'  => $request->sabha_id,
                'date'      => $request->date ?? $attendance->date,
            ]);

            return response()->json([
                'message' => 'Attendance updated successfully',
                'attendance' => $attendance
            ], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Server error',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // DELETE: Remove an attendance record
    public function destroy($id)
    {
        try {
            $attendance = Attendance::find($id);

            if (!$attendance) {
                return response()->json(['message' => 'Attendance not found'], 404);
            }

            $attendance->delete();

            return response()->json(['message' => 'Attendance deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Server error',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function markAttendance(Request $request)
    {
        try {
            $request->validate([
                'member_id' => 'required|exists:members,id',
                'sabha_id'  => 'required|exists:sabhas,id',
            ]);

            $today = now()->toDateString();

            $attendance = Attendance::where('member_id', $request->member_id)
                ->where('sabha_id', $request->sabha_id)
                ->whereDate('date', $today)
                ->first();

            if ($attendance) {
                return response()->json([
                    'message' => 'Attendance already marked for this Sabha',
                    'attendance' => $attendance
                ], 200);
            }

            $newAttendance = Attendance::create([
                'member_id' => $request->member_id,
                'sabha_id'  => $request->sabha_id,
                'date' => $today,
            ]);

            return response()->json([
                'message' => 'Attendance marked successfully',
                'attendance' => $newAttendance
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Server error',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
