<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use TCG\Voyager\Models\Member;
use Illuminate\Http\Request;
use PDF;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Http;

class MemberController extends Controller
{
    // GET: Fetch all members
    public function index()
    {
        $members = Member::all();
        return response()->json($members);
    }

    // POST: Add a new member
    public function store(Request $request)
    {
        try {
            $request->validate([
                'first_name' => 'required|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'member_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
                'number' => 'required|string|regex:/^\d{10}$/',
                'gender' => 'nullable|string|max:20',
                'dob' => 'nullable|date',
                'age' => 'nullable|string|max:20',
                'address' => 'nullable|string',
                'city' => 'nullable|string',
                'mandal' => 'nullable|integer',
                'status' => 'nullable|integer',
                'Karyakarta' => 'nullable|string|max:255',
            ], [
                'first_name.required' => 'The first name field is required.',
                'number.required' => 'The phone number field is required.'
            ]);
    
            // Prepare data for insertion
            $memberData = $request->all();
    
            // Ensure the number starts with +91
            $phone_number = $request->input('number');
            if (substr($phone_number, 0, 3) !== '+91') {
                $phone_number = '+91' . $phone_number;
            }
            $memberData['number'] = $phone_number;  // Update the number in member data
    
            // Check if image is present
            if ($request->hasFile('member_image')) {
                // Handle the image upload
                $imagePath = $request->file('member_image')->store('member_images', 'public');
                $memberData['member_image'] = $imagePath;
            } else {
                // Default image if none is provided
                $memberData['member_image'] = 'member_images/default.png';
            }
    
            // Create member
            $member = Member::create($memberData);
    
            // Generate the membership card PDF
            // $pdf = PDF::loadView('admin.members.card', compact('member'));
            $pdf = Pdf::loadView('pdf.card', compact('member'));

            // Store the PDF in a public directory
            $pdfPath = 'pdfs/membership_card_' . $member->id . '.pdf';
            Storage::disk('public')->put($pdfPath, $pdf->output());

            // Generate the URL for the PDF file
            $pdfUrl = Storage::disk('public')->url($pdfPath);
    
            // Send the PDF via WhatsApp
            $this->sendCardOnWhatsapp($member->number, $pdfUrl);

            return response()->json(['message' => 'Member created successfully and card sent!', 'member' => $member], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
    }

    // UPDATE: Update an existing member
    public function update(Request $request, $id)
    {
        try {
            $member = Member::find($id);
    
            if (!$member) {
                return response()->json(['message' => 'Member not found'], 404);
            }
    
            // Validation for the request
            $request->validate([
                'first_name' => 'nullable|string|max:255',
                'last_name' => 'nullable|string|max:255',
                'member_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
                'number' => 'nullable|string|regex:/^\d{10}$/',
                'gender' => 'nullable|string|max:20',
                'dob' => 'nullable|date',
                'age' => 'nullable|string|max:20',
                'address' => 'nullable|string',
                'city' => 'nullable|string',
                'mandal' => 'nullable|integer',
                'status' => 'nullable|integer',
                'Karyakarta' => 'nullable|string|max:255',
            ]);
    
            // Handle the phone number to add +91 if not present
            if ($request->has('number')) {
                $phone_number = $request->input('number');
                if (substr($phone_number, 0, 3) !== '+91') {
                    $phone_number = '+91' . $phone_number;
                }
                $request->merge(['number' => $phone_number]);
            }
    
            // Update member data with new values (excluding member_image)
            $member->update($request->except('member_image'));
    
            // Handle file upload if a new image is provided
            if ($request->hasFile('member_image')) {
                $imagePath = $request->file('member_image')->store('member_images', 'public'); // Store image in `storage/app/public/member_images`
                $member->member_image = $imagePath;  // Save the new image path in the member record
                $member->save(); // **Save the model to persist the image path**
            }
    
            // Generate the membership card PDF
            $pdf = Pdf::loadView('pdf.card', compact('member'));

            // Store the PDF in a public directory
            $pdfPath = 'pdfs/membership_card_' . $member->id . '.pdf';
            Storage::disk('public')->put($pdfPath, $pdf->output());

            // Generate the URL for the PDF file
            $pdfUrl = Storage::disk('public')->url($pdfPath);
    
            // Send the PDF via WhatsApp
            $this->sendCardOnWhatsapp($member->number, $pdfUrl);

            return response()->json(['message' => 'Member updated successfully and card sent!', 'member' => $member]);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json(['message' => 'An unexpected error occurred.'], 500);
        }
    }

    // Helper function to send card on WhatsApp
    private function sendCardOnWhatsapp($number, $pdfUrl)
    {
        $response = Http::post('http://wa.rigelpay.in/api/v1/sendDocument', [
            'key' => 'e4613dac2c8e428e9b327cd79c61669a',
            'to' => $number,
            'url' => $pdfUrl,
            'filename' => 'membership_card.pdf'
        ]);

        $responseData = $response->json();

        if ($response->successful() && isset($responseData['ErrorCode']) && $responseData['ErrorCode'] === '000') {
            \Log::info('WhatsApp document sent', ['member_id' => $number, 'api_reference' => $responseData['Data'] ?? null]);
        } else {
            \Log::error('WhatsApp API error', ['member_id' => $number, 'error_code' => $responseData['ErrorCode'] ?? 'Unknown', 'error_message' => $responseData['ErrorMessage'] ?? 'No message returned']);
        }
    }

    // DELETE: Delete a member
    public function destroy($id)
    {
        $member = Member::find($id);

        if (!$member) {
            return response()->json(['message' => 'Member not found'], 404);
        }

        $member->delete();

        return response()->json(['message' => 'Member deleted successfully']);
    }

    public function generateCard($id)
    {
        $member = Member::findOrFail($id);
        return view('admin.members.card', compact('member'));
    }

    // Helper function to save Base64 image
    private function saveBase64Image($base64String)
    {
        // Validate Base64 format
        if (!preg_match('/^data:image\/(\w+);base64,/', $base64String, $matches)) {
            return null;
        }

        $imageType = $matches[1]; // jpg, png, gif, etc.
        $imageData = substr($base64String, strpos($base64String, ',') + 1);
        $imageData = base64_decode($imageData);

        // Generate unique filename
        $filename = 'member_images/' . Str::random(20) . '.' . $imageType;

        // Store in Laravel storage (storage/app/public/member_images)
        Storage::disk('public')->put($filename, $imageData);

        return $filename; // Save path in database
    }

    public function getAllMemberCards()
    {
        $members = Member::all();

        $data = $members->map(function ($member) {
            $fileName = "membership_card_{$member->id}.pdf";
            $filePath = public_path("storage/pdfs/{$fileName}");

            return [
                'id' => $member->id,
                'membership_card' => file_exists($filePath)
                    ? asset("storage/pdfs/{$fileName}")
                    : null,
            ];
        });

        return response()->json($data);
    }
}
